/** Datentyp fuer Suchbaeume mit Objekten, welche zu einer Klasse
 *  gehoeren, die sowohl Ord als auch Vergleichbar entspricht
 * @see Knoten
 * @see VergleichbarUndOrd
 * @author Antje Nowack
 */


/* Umgebung : JDK 1.3, Linux
 * Erstellt : 28. 12.2001 
 * Letzte Aenderung: 7. 1. 2002
 */

public class Suchbaum {
    
    /** Wurzel des Suchbaums
     */
    private Knoten wurzel;
    
    
    /** erzeugt eine neue leere Suchbaum
     */
    public Suchbaum () {
	wurzel = null;
    }


    /** liefert den linken Teilbaum
     *  @return linker Teilbaum
     */
    public Suchbaum linkerTeilbaum () {
      Suchbaum h = new Suchbaum ();
      h.wurzel = wurzel.getLeft();
      return h;
    }


    /** liefert den rechten Teilbaum
     *  @return linker Teilbaum
     */
    public Suchbaum RechterTeilbaum () {
      Suchbaum h = new Suchbaum ();
      h.wurzel = wurzel.getRight();
      return h;
    }

    
    /** sucht nach einem Knoten im Suchbaum.
     *   @param wert Der Wert des gesuchten Knotens.
     *   @return Der erste Knoten im Suchbaum mit diesem Wert, falls
     *           es ein solches gibt. Sonst wird null zurckgegeben.
     */
    public Knoten suche (VergleichbarUndOrd wert) {
	return suche (wert, wurzel);
    }
    
    /** sucht nach einem Knoten in einem vorgegebenen Suchbaum.
     *   @param wert Der Wert des gesuchten Knotens.
     *   @param wurzel Die Wurzel des Suchbaums, in dem gesucht wird.
     *   @return Der erste Knoten in jenem Suchbaum mit diesem Wert, falls
     *           es einen solches gibt. Sonst wird null zurckgegeben.
     */
    private static Knoten suche (VergleichbarUndOrd wert, Knoten wurzel) {
	if	(wurzel == null)  return null;
	else if (wert.gleich(wurzel.getWert())) return wurzel;
	else {
	    // rekursiver Aufruf mit linkem Teilbaum, da das der Wert, wenn
            // er enthalten ist, in diesem enthalten sein muss
          if (suche (wert, wurzel.getLeft()) == null) 
            return suche (wert, wurzel.getRight());
          else
            // rekursiver Aufruf mit rechtem Teilbaum
            return suche (wert, wurzel.getRight());
        }
    }
    
    /** erzeugt einen String, der die Knoten des Suchbaums in Infixnotation
     *   darstellt.
     *   @return Suchbaum in Infixnotation.
     */
    public String toString () {
	return 	"( " + durchlaufeInfix(wurzel) + ")";
    }
    
    /** gibt den Inhalt des Suchbaums (in Infixnotation) auf dem Bildschirm 
     *  aus.
     *  @return Suchbaum als Zeichenkette in Infixnotation.
     */
    public void drucke() {
	System.out.println (this);
    }
    
    
    /** erzeugt einen String, der aus allen Knoten eines vorgegebenen 
     *  Suchbaums (in Infixnotation) besteht.
     *  @param wurzel Die Wurzel des zu durchlaufenden Suchbaum.
     *  @return Die Zeichenkette aller Knoten jenes Suchbaums (in 
     *          Infixnotation).
     */
    private static String durchlaufeInfix (Knoten k) {        
	if (!(k == null)) {
            // Ausgabe linker Teilbaum durch rekursiven Aufruf, dann Wurzel,
            // dann rechter Teilbaum, wobei die Einteilung durch Klammern
            // dargewstelt wird
            return "(" + durchlaufeInfix(k.getLeft()) + " " + k.getWert() + " " + durchlaufeInfix(k.getRight()) + ")";
        }
	else return "";
    }
    

    /** fuegt Wert in einen vorgegebenen Suchbaum ein
     *  @param n einzufuegender Wert
     *  @param wurzel Wurzel des Suchbaum, in den eingefuegt werden soll
     *  @return Wurzel des resulktierenden Suchbaums      
     */
   private static Knoten fuegeEin (VergleichbarUndOrd n, Knoten k) {
     if (k == null) {
         k = new Knoten (n);
         return k;          
     }
     else {
        if ((k.getWert().groesser (n)) | (k.getWert().gleich (n))) {
	    // der Wert wird in den linken Teilbaum eingefuegt, da kleiner
	    // als die Wurzel oder gleich
          return new Knoten (k.getWert(), fuegeEin(n, k.getLeft()), k.getRight());
        }
        else {
	    // der Wert wird in den rechten Teilbaum eingefuegt 
          return new Knoten (k.getWert(), k.getLeft(), fuegeEin(n, k.getRight()));
        } 
     }
   }
 
   /** fuegt den uebergebenen Wert in den aktuellen Suchbaum ein
     *  @param n einzufuegender Wert
     */
  void fuegeEin(VergleichbarUndOrd k) {
     wurzel = fuegeEin (k, wurzel);
  }	   			
			
}	

